const { app, BrowserWindow, ipcMain } = require("electron");
const { spawn, spawnSync } = require("child_process");
const fs = require("fs");
const path = require("path");
const os = require("os");
const https = require("https");
const http = require("http");
const unzipper = require("unzipper");
const { pipeline } = require("stream/promises");

const BASE_URL = "http://2hme.web.svpj.pl/launcher";
const HOME = app.getPath("home");

const ROOT = path.join(HOME, ".gascraft");
const JAR = path.join(ROOT, "gascraft.jar");
const NATIVES = path.join(ROOT, "natives");
const JRE = path.join(ROOT, "jre");

let mainWindow;

/* ===================== UI COMM ===================== */

function send(msg, data = {}) {
  if (mainWindow) mainWindow.webContents.send("status", { msg, ...data });
  console.log(msg, data);
}

/* ===================== FS UTILS ===================== */

function rmrf(p) {
  try {
    fs.rmSync(p, { recursive: true, force: true });
  } catch {}
}

function mkdirp(p) {
  fs.mkdirSync(p, { recursive: true });
}

function exists(p) {
  try {
    fs.accessSync(p);
    return true;
  } catch {
    return false;
  }
}

function fileSize(p) {
  try {
    return fs.statSync(p).size;
  } catch {
    return 0;
  }
}

/* ===================== HTTP DOWNLOAD (ROBUST) ===================== */

function getClient(url) {
  return url.startsWith("https") ? https : http;
}

function requestOnce(url) {
  return new Promise((resolve, reject) => {
    const client = getClient(url);
    const req = client.get(url, (res) => resolve(res));
    req.on("error", reject);
  });
}

async function download(url, dest, label, maxRedirects = 5) {
  mkdirp(path.dirname(dest));

  const part = dest + ".part";
  rmrf(part);

  let current = url;

  for (let i = 0; i <= maxRedirects; i++) {
    const res = await requestOnce(current);

    // redirect
    if (res.statusCode >= 300 && res.statusCode < 400 && res.headers.location) {
      const next = new URL(res.headers.location, current).toString();
      res.resume();
      current = next;
      continue;
    }

    if (res.statusCode !== 200) {
      res.resume();
      throw new Error(`HTTP ${res.statusCode}`);
    }

    const total = parseInt(res.headers["content-length"] || "0", 10);
    let downloaded = 0;

    res.on("data", (chunk) => {
      downloaded += chunk.length;
      if (total > 0) {
        const progress = Math.min(99, Math.floor((downloaded / total) * 100));
        send(label, { progress, downloaded, total });
      } else {
        // gdy brak content-length: pokazuj “żyje” bez procentów
        send(label, { progress: null, downloaded, total: null });
      }
    });

    const out = fs.createWriteStream(part);

    try {
      await pipeline(res, out);
    } catch (e) {
      rmrf(part);
      throw e;
    }

    // Walidacja rozmiaru jeżeli serwer podał content-length
    if (total > 0) {
      const got = fileSize(part);
      if (got !== total) {
        rmrf(part);
        throw new Error(`Download incomplete (${got}/${total})`);
      }
    }

    // atomowy rename
    rmrf(dest);
    fs.renameSync(part, dest);

    send(label, { progress: 100, downloaded: total || fileSize(dest), total: total || fileSize(dest) });
    return;
  }

  throw new Error("Too many redirects");
}

async function downloadWithRetry(url, dest, label, tries = 2) {
  let lastErr = null;
  for (let attempt = 1; attempt <= tries; attempt++) {
    try {
      if (attempt > 1) send(label, { progress: 0, retry: attempt });
      await download(url, dest, label);
      return;
    } catch (e) {
      lastErr = e;
      rmrf(dest);
      rmrf(dest + ".part");
    }
  }
  throw lastErr || new Error("Download failed");
}

/* ===================== UNZIP (ROBUST) ===================== */

async function unzip(zipPath, targetDir) {
  if (!exists(zipPath) || fileSize(zipPath) < 1024) {
    throw new Error("ZIP is missing or too small");
  }

  mkdirp(targetDir);

  // WINDOWS – tylko PowerShell
  if (process.platform === "win32") {
    const ps = spawnSync(
      "powershell",
      [
        "-NoProfile",
        "-Command",
        `Expand-Archive -LiteralPath "${zipPath}" -DestinationPath "${targetDir}" -Force`
      ],
      { stdio: "ignore" }
    );

    if (ps.status !== 0) {
      throw new Error("Expand-Archive failed");
    }
    return;
  }

  // macOS / Linux – unzipper + fallback
  try {
    await fs.createReadStream(zipPath)
      .pipe(unzipper.Extract({ path: targetDir }))
      .promise();
    return;
  } catch {}

  if (process.platform === "darwin") {
    const ditto = spawnSync("ditto", ["-xk", zipPath, targetDir], { stdio: "ignore" });
    if (ditto.status === 0) return;

    const unzipCmd = spawnSync("unzip", ["-o", zipPath, "-d", targetDir], { stdio: "ignore" });
    if (unzipCmd.status === 0) return;
  }

  throw new Error("Unzip failed");
}


/* ===================== macOS HELPERS ===================== */

function resolveMacJava() {
  const visited = new Set();
  const stack = [JRE];

  while (stack.length) {
    const current = stack.pop();
    if (visited.has(current)) continue;
    visited.add(current);

    const candidateWithContents = path.join(current, "Contents", "Home", "bin", "java");
    if (exists(candidateWithContents)) return candidateWithContents;

    const candidateBin = path.join(current, "bin", "java");
    if (exists(candidateBin)) return candidateBin;

    let entries;
    try {
      entries = fs.readdirSync(current, { withFileTypes: true });
    } catch {
      continue;
    }

    for (const entry of entries) {
      if (entry.isDirectory()) {
        stack.push(path.join(current, entry.name));
      } else if (entry.isSymbolicLink()) {
        try {
          const resolved = fs.realpathSync(path.join(current, entry.name));
          if (resolved.startsWith(JRE)) stack.push(resolved);
        } catch {}
      }
    }
  }

  throw new Error("Nie znaleziono runtime po rozpakowaniu");
}

function isExecutable(file) {
  try {
    fs.accessSync(file, fs.constants.X_OK);
    return true;
  } catch {
    return false;
  }
}

function clearQuarantineSync(target) {
  // sync żeby nie było wyścigu przed spawn(java)
  spawnSync("xattr", ["-dr", "com.apple.quarantine", target], { stdio: "ignore" });
}

function chmodRecursive(dir) {
  try { fs.chmodSync(dir, 0o755); } catch {}
  let entries;
  try { entries = fs.readdirSync(dir, { withFileTypes: true }); } catch { return; }
  for (const e of entries) {
    const p = path.join(dir, e.name);
    if (e.isDirectory()) chmodRecursive(p);
    else {
      try { fs.chmodSync(p, 0o755); } catch {}
    }
  }
}

function repairMacRuntimePermissions() {
  if (!exists(JRE)) return null;

  try { clearQuarantineSync(JRE); } catch {}
  try { chmodRecursive(JRE); } catch {}

  try {
    const java = resolveMacJava();
    try { fs.chmodSync(java, 0o755); } catch {}
    if (!isExecutable(java)) return null;
    return java;
  } catch {
    return null;
  }
}
function resolveWindowsJava() {
  const candidates = [];

  try {
    const entries = fs.readdirSync(JRE, { withFileTypes: true });
    for (const e of entries) {
      if (e.isDirectory()) {
        candidates.push(path.join(JRE, e.name, "bin", "java.exe"));
      }
    }
  } catch {}

  for (const c of candidates) {
    if (exists(c)) return c;
  }

  throw new Error("Nie znaleziono java.exe w JRE/JDK");
}

/* ===================== PICK FILES ===================== */

function pickRuntimeZipName() {
  // UWAGA: teraz pobieramy JDK, nie JRE
  if (process.platform === "win32") return "jdk-22-windows-x64.zip";
  if (process.platform === "darwin") {
    return os.arch() === "arm64" ? "jdk-22-macos-arm64.zip" : "jdk-22-macos-x64.zip";
  }
  throw new Error("Unsupported OS");
}

function pickNativesZipName() {
  if (process.platform === "win32") return "natives-windows.zip";
  return os.arch() === "arm64" ? "natives-macos-arm64.zip" : "natives-macos.zip";
}

/* ===================== ENSURE RUNTIME ===================== */

async function ensureJRE() {
  mkdirp(ROOT);

  // jeśli jest już java i jest wykonywalna → OK (na macOS i tak potem naprawimy perms)
  if (process.platform === "darwin") {
    try {
      const javaBin = resolveMacJava();
      if (exists(javaBin) && isExecutable(javaBin)) return;

      // jeśli istnieje, ale nie jest wykonywalna -> spróbuj naprawić
      if (exists(javaBin)) {
        send("Naprawianie uprawnień JRE");
        const repaired = repairMacRuntimePermissions();
        if (repaired && isExecutable(repaired)) return;
      }
    } catch {}
  } else {
    const javaBin = path.join(JRE, "bin", process.platform === "win32" ? "java.exe" : "java");
    if (exists(javaBin)) return;
  }

  // Pobieramy na czysto: usuń stare runtime (żeby nie mieszać)
  rmrf(JRE);
  mkdirp(JRE);

  const runtimeZip = pickRuntimeZipName();
  const zipPath = path.join(ROOT, runtimeZip);

  send("Pobieranie JRE", { progress: 0 });
  await downloadWithRetry(`${BASE_URL}/${runtimeZip}`, zipPath, "Pobieranie JRE", 2);

  // Rozpakuj: jeśli zip jest ucięty -> retry pobrania
  send("Rozpakowywanie JRE");
  try {
    await unzip(zipPath, JRE);
  } catch (e) {
    // usuń i spróbuj jeszcze raz pobrać (częsty przypadek gdy zip “niby 100%” a jest ucięty)
    rmrf(JRE);
    mkdirp(JRE);
    rmrf(zipPath);
    send("Pobieranie JRE", { progress: 0, retry: 2 });
    await downloadWithRetry(`${BASE_URL}/${runtimeZip}`, zipPath, "Pobieranie JRE", 2);
    send("Rozpakowywanie JRE");
    await unzip(zipPath, JRE);
  }

  if (process.platform === "darwin") {
    send("Naprawianie uprawnień JRE");
    const repaired = repairMacRuntimePermissions();
    if (!repaired) throw new Error("Nie udało się przygotować java na macOS (chmod/xattr)");
  }
}

/* ===================== ENSURE GAME ===================== */

async function ensureGame() {
  mkdirp(ROOT);
  mkdirp(NATIVES);

  if (!exists(JAR) || fileSize(JAR) < 1024) {
    rmrf(JAR);
    send("Pobieranie gry", { progress: 0 });
    await downloadWithRetry(`${BASE_URL}/gascraft.jar`, JAR, "Pobieranie gry", 2);
  }

  const nativesZip = pickNativesZipName();
  const zipPath = path.join(ROOT, nativesZip);

  // jeśli natives folder pusty / brak → rozpakuj ponownie
  const nativesLooksOk = (() => {
    try {
      const items = fs.readdirSync(NATIVES);
      return items.length > 0;
    } catch {
      return false;
    }
  })();

  if (!nativesLooksOk) {
    rmrf(NATIVES);
    mkdirp(NATIVES);

    send("Pobieranie natives", { progress: 0 });
    await downloadWithRetry(`${BASE_URL}/${nativesZip}`, zipPath, "Pobieranie natives", 2);

    send("Rozpakowywanie natives");
    try {
      await unzip(zipPath, NATIVES);
    } catch (e) {
      rmrf(NATIVES);
      mkdirp(NATIVES);
      rmrf(zipPath);
      send("Pobieranie natives", { progress: 0, retry: 2 });
      await downloadWithRetry(`${BASE_URL}/${nativesZip}`, zipPath, "Pobieranie natives", 2);
      send("Rozpakowywanie natives");
      await unzip(zipPath, NATIVES);
    }
  }
}

/* ===================== LAUNCH ===================== */

function launchGame() {
  send("Uruchamianie gry");

  let javaBin;
  if (process.platform === "darwin") {
    javaBin = resolveMacJava();
  } else if (process.platform === "win32") {
    javaBin = resolveWindowsJava();
  } else {
    javaBin = path.join(JRE, "bin", "java");
  }

  const args = [];
  if (process.platform === "darwin") args.push("-XstartOnFirstThread");

  args.push(
    `-Djava.library.path=${NATIVES}`,
    "-jar",
    JAR
  );

  console.log("JAVA:", javaBin);
  console.log("ARGS:", args);

  const child = spawn(javaBin, args, { stdio: "inherit" });

  child.on("error", (e) => {
    send("Błąd", { error: e.message });
  });

  child.on("exit", (code) => {
    if (code !== 0) {
      send("Błąd", { error: `Java zakończyła się kodem ${code}` });
    }
  });
}

/* ===================== IPC ===================== */

ipcMain.handle("play", async () => {
  try {
    await ensureJRE();
    await ensureGame();
    launchGame();
  } catch (e) {
    send("Błąd", { error: e.message });
  }
});

/* ===================== APP ===================== */

app.whenReady().then(() => {
  mainWindow = new BrowserWindow({
    width: 800,
    height: 400,
    resizable: false,
    webPreferences: {
      preload: path.join(__dirname, "preload.js")
    }
  });

  mainWindow.loadFile("index.html");
});
